"""
保证金模块 margin 的使用方法及期货、期权（商品、股指、ETF）开仓预算保证金示例，仅供测试，请勿直接使用。
margin_ratios.json 取自无限易模拟全部期货品种的主力合约保证金率作为品种保证金率，仅作示例。后续请自行维护。

更新时间：2025-12-10
"""

from pathlib import Path
from typing import Any, Literal, Optional

from pythongo.base import BaseParams, BaseState, BaseStrategy, Field
from pythongo.classdef import TickData, TradeData
from pythongo.types import TypeHedgeFlag, TypeOrderFlag

from self_strategy.margin import (
    MarginRateConfig,
    MarginCalculator,
    PrePriceData,
)


class Params(BaseParams):
    """参数映射模型"""

    exchange: str = Field(default="", title="交易所代码")
    instrument_id: str = Field(default="", title="合约代码")
    order_volume: int = Field(default=1, title="下单手数", ge=1)
    order_direction: Literal["buy", "sell"] = Field(default="sell", title="下单方向")
    investor: str = Field(default="", title="下单账户")


class State(BaseState):
    """状态映射模型"""

    pass


class 开仓预算保证金示例(BaseStrategy):
    """期货、期权开仓预算保证金示例 - 仅供测试"""

    def __init__(self) -> None:
        super().__init__()

        self.params_map = Params()
        """参数表"""

        self.underlying_exchange: str = ""
        """标的交易所代码"""

        self.underlying_symbol: str = ""
        """标的合约代码"""

        self.pre_data: PrePriceData = PrePriceData()
        """当前合约及标的昨收、昨结数据"""

        self.margin_calculator: Optional[MarginCalculator] = None
        """保证金计算器实例"""

    def on_tick(self, tick: TickData) -> None:
        """收到行情 tick 推送"""

        # 更新当前合约行情
        if tick.instrument_id == self.params_map.instrument_id:
            self.pre_data.pre_settlement = tick.pre_settlement_price

            # 判断是否满足下单条件
            should_order = False

            # 期货下单 (无标的)
            if self.trading and not self.underlying_symbol:
                should_order = True

            # 期权下单 (有标的且已获取标的昨结价)
            elif (
                self.trading
                and self.underlying_symbol
                and self.pre_data.underlying_pre_close
            ):
                should_order = True

            if should_order:
                self.check_margin_send_order(
                    exchange=self.params_map.exchange,
                    instrument_id=self.params_map.instrument_id,
                    volume=self.params_map.order_volume,
                    price=(
                        tick.ask_price1
                        if self.params_map.order_direction == "buy"
                        else tick.bid_price1
                    ),
                    order_direction=self.params_map.order_direction,
                    investor=self.params_map.investor,
                )
                self.trading = False  # 只下单一次用于测试

        # 获取标的昨收、昨结价
        elif self.underlying_symbol and tick.instrument_id == self.underlying_symbol:
            self.pre_data.underlying_pre_close = tick.pre_close_price
            self.pre_data.underlying_pre_settlement = tick.pre_settlement_price

            # 取到标的昨结、昨收后取消订阅
            self.unsub_market_data(
                exchange=self.underlying_exchange,
                instrument_id=self.underlying_symbol,
            )

        super().on_tick(tick)

    def on_trade(self, trade: TradeData, log: bool = False) -> None:
        """成交回调"""
        super().on_trade(trade, log)

    def on_start(self) -> None:
        """策略启动回调"""

        # 把 margin_ratios.json 放到策略所在文件夹，拼接出 json 文件的绝对路径
        current_folder = Path(__file__).parent
        config_path = current_folder / "margin_ratios.json"
        self.margin_calculator = MarginCalculator(config_path=str(config_path))

        # =========================================================
        # [示例代码] 保证金管理器使用演示
        # 注意：如需查看效果，取消注释下面代码块
        # =========================================================
        """

        # --- 场景 A: 获取管理器实例（基于 json 文件）
        mgr = self.margin_calculator.margin_ratios

        # 1. 查询费率 (get_ratio)
        # 作用：根据 合约代码 -> 品种代码 -> 默认值 的顺序查找
        # 示例：查找 rb2605，如果 json 里没有 rb2605，就会找 rb，再找不到就返 0.12
        ratio = mgr.get_ratio(
            exchange="SHFE", instrument_id="rb2605", default_ratio=0.12
        )
        self.output(f"[get_ratio] 查询 rb2605 保证金率: {ratio:.2%}")

        # 2. 单个更新 (set_ratio)
        # 作用：修改单个合约/品种的保证金率
        # 示例：假设交易所临时提高 rb2605 的保证金到 20%
        mgr.set_ratio(target_id="rb2605", new_ratio=0.20)

        # 3. 批量更新 (update_ratios)
        # 作用：一次性更新多个品种或合约的保证金率
        # 示例：同时更新 白银(ag) 和 黄金(au)
        updates = {"ag": 0.15, "au": 0.09}  # 提高到 15%  # 设为 9%
        mgr.update_ratios(updates)

        # 4. 保存配置 (save)
        # 作用：将当前内存中所有修改后的配置，写入到硬盘文件
        # 示例：保存到当前目录下的 margin_ratios_new.json
        backup_path = current_folder / "margin_ratios_new.json"
        mgr.save()  # 覆盖原文件
        mgr.save(filepath=str(backup_path))  # 另存为新文件

        # 5. 从文件加载 (load_from_json)
        # 作用：读取外部 json 文件并覆盖/更新当前配置
        # 示例：重新从刚刚保存的新配置文件中读取数据
        mgr.load_from_json(filepath=backup_path)
        self.output(f"[load_from_json] 已重新从新配置文件加载配置")
        
        # --- 场景 B：不读取文件，纯内存模式
        self.margin_calculator = MarginCalculator(
            default_ratio=0.12
        )  # 取不到保证金时用默认保证金率 0.12
        mgr = self.margin_calculator.margin_ratios
        mgr.update_ratios({"AP": 0.1, "AP512": 0.2})
        self.output(f"默认配置：{mgr._ratios}")
        # 其他方法与场景 A 一样

        """

        instrument_info = self.margin_calculator._instrument_data(
            self.params_map.exchange, self.params_map.instrument_id
        )

        if instrument_info.underlying_symbol:
            # 如果是期权合约需要订阅标的行情
            self.underlying_symbol = instrument_info.underlying_symbol
            self.underlying_exchange = instrument_info.underlying_exchange
            self.sub_market_data(
                exchange=self.underlying_exchange,
                instrument_id=self.underlying_symbol,
            )

        super().on_start()

    def on_stop(self) -> None:
        """策略停止回调"""

        if self.underlying_exchange and self.underlying_symbol:
            self.unsub_market_data(
                exchange=self.underlying_exchange, instrument_id=self.underlying_symbol
            )
        self.underlying_exchange = ""
        self.underlying_symbol = ""
        self.pre_data = PrePriceData()

        super().on_stop()

    def check_margin_send_order(
        self,
        exchange: str,
        instrument_id: str,
        volume: int,
        price: float | int,
        order_direction: Literal["buy", "sell"],
        order_type: TypeOrderFlag = "GFD",
        investor: str = "",
        hedgeflag: TypeHedgeFlag = "1",
        market: bool = False,
        memo: Any = None,
    ) -> int | None:
        """预算保证金/资金占用的开仓报单函数"""

        if not investor:
            investor = self.get_investor_data().investor_id
        fund_data = self.get_account_fund_data(investor=investor)
        available = fund_data.available

        margin = self.margin_calculator.calculate_order_margin(
            exchange=exchange,
            instrument_id=instrument_id,
            price=price,
            volume=volume,
            order_direction=order_direction,
            pre_data=self.pre_data,
        )
        if margin is None:
            self.output(f"无法计算保证金，取消下单！")
            return

        if available < margin:
            self.output(f"当前可用资金: {available} < 预算保证金: {margin}，取消下单！")
            return

        self.output(f"当前可用资金: {available} >= 预算保证: {margin}，准备下单！")
        return self.send_order(
            exchange=exchange,
            instrument_id=instrument_id,
            volume=volume,
            price=price,
            order_direction=order_direction,
            order_type=order_type,
            investor=investor,
            hedgeflag=hedgeflag,
            market=market,
            memo=memo,
        )
